<?php
/*
 * @package   bfNetwork
 * @copyright Copyright (C) 2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023 Blue Flame Digital Solutions Ltd. All rights reserved.
 * @license   GNU General Public License version 3 or later
 *
 * @see       https://mySites.guru/
 * @see       https://www.phil-taylor.com/
 *
 * @author    Phil Taylor / Blue Flame Digital Solutions Limited.
 *
 * bfNetwork is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * bfNetwork is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this package.  If not, see http://www.gnu.org/licenses/
 *
 * If you have any questions regarding this code, please contact phil@phil-taylor.com
 */

use Akeeba\Engine\Platform;
use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Factory;

require 'bfEncrypt.php';

/**
 * If we have got here then we have already passed through decrypting the encrypted header and so we are sure we are now
 * secure and no one else cannot run the code below.
 */
final class bfBackup
{
    /**
     * We pass the command to run as a simple integer in our encrypted request this is mainly to speed up the decryption
     * process, plus its a single digit(or 2) rather than a huge string to remember :-).
     */
    private array $_methods = [
        1 => 'enableAkeebaFrontendBackup',
    ];

    /**
     * Pointer to the Joomla Database Object.
     */
    private $_db;

    /**
     * I inject the request to the object.
     *
     * @param stdClass $_dataObj
     */
    public function __construct(
        /**
         * Incoming decrypted vars from the request.
         */
        private $_dataObj,
    ) {
        // init Joomla
        if (! defined('BF_JOOMLA_INIT_DONE')) {
            require_once 'bfInitJoomla.php';
        }
    }

    /**
     * I'm the controller - I run methods based on the request integer.
     */
    public function run()
    {
        if (property_exists($this->_dataObj, 'c')) {
            $c = (int) $this->_dataObj->c;
            if (array_key_exists($c, $this->_methods)) {
                // call the right method
                $this->{$this->_methods[$c]} ();
            } else {
                // Die if an unknown function
                bfEncrypt::reply('error', 'No Such method #err1 - ' . $c);
            }
        } else {
            // Die if an unknown function
            bfEncrypt::reply('error', 'No Such method #err2');
        }
    }

    /**
     * If not enabled, then enable the Akeeba API Frontend using a secure secret word.
     */
    private function enableAkeebaFrontendBackup()
    {
        // load mini-Joomla
        if (! defined('BF_JOOMLA_INIT_DONE')) {
            require_once 'bfInitJoomla.php';
        }

        $this->_db = Factory::getContainer()->get('DatabaseDriver');

        $this->_db->setQuery('SELECT extension_id, params FROM #__extensions WHERE element like "com_akeeba%"');
        $data = $this->_db->loadObject();

        if (! $data) {
            // send back the totals
            bfEncrypt::reply('success', [
                '__LINE__'         => __LINE__,
                'akeeba_installed' => false,
            ]);
        }

        $params = json_decode((string) $data->params, true, 512, JSON_THROW_ON_ERROR);

        if (! $params) {
            bfEncrypt::reply('success', [
                '__LINE__'         => __LINE__,
                'akeeba_installed' => false,
            ]);
        }

        if (! \count($params)) {
            // means only just installed Akeeba and params is currently {}
        }

        // Get a complex unique non-crypto string from mySites.guru
        $secretWord                = $params['frontend_secret_word'] = file_get_contents(
            'https://manage.mysites.guru/public/rand?' . time()
        );
        $params['jsonapi_enabled'] = 1;

        // Encrypt the frontend secret the Akeeba Way.
        $this->initAkeeba();
        $secureSettings                 = \Akeeba\Engine\Factory::getSecureSettings();
        $params['frontend_secret_word'] = $secureSettings->encryptSettings($params['frontend_secret_word']);

        $sql = 'UPDATE #__extensions SET params = \'%s\' WHERE extension_id = %s';
        $sql = sprintf($sql, addslashes(json_encode($params, JSON_THROW_ON_ERROR)), $data->extension_id);
        $this->_db->setQuery($sql);
        $this->_db->execute();

        $cache = Factory::getContainer()
            ->get(CacheControllerFactoryInterface::class)
            ->createCacheController('callback', [
                'defaultgroup' => '_system',
            ]);
        $cache->clean();

        bfEncrypt::reply('success', [
            'akeeba_installed' => true,
            'secret'           => $secretWord,
        ]);
    }

    private function initAkeeba()
    {
        if (! defined('AKEEBAENGINE')) {
            // Load Composer dependencies
            $autoloader = require_once JPATH_ADMINISTRATOR . '/components/com_akeebabackup/vendor/autoload.php';

            // Defines for Akeeba Engine
            define('AKEEBAENGINE', 1);
            define('AKEEBAROOT', JPATH_ADMINISTRATOR . '/components/com_akeebabackup/vendor/akeeba/engine/engine');

            include AKEEBAROOT . '/Factory.php';

            Platform::addPlatform('joomla', JPATH_ADMINISTRATOR . '/components/com_akeebabackup/platform/Joomla');

            // Apply a custom path for the encrypted settings key file
            \Akeeba\Engine\Factory::getSecureSettings()->setKeyFilename(
                JPATH_ADMINISTRATOR . '/components/com_akeebabackup/serverkey.php',
            );
        }
    }
}

/** @var stdClass $dataObj The incoming decrypted request */
$backupController = new bfBackup($dataObj);

// Run the tool method
$backupController->run();
